/**
 * define SwitcherManageOptions
 *
 ***/
cr.define('options', function() {
  var OptionsPage = options.OptionsPage;
  var InlineEditableItemList = options.InlineEditableItemList;
  var InlineEditableItem = options.InlineEditableItem;
  var ArrayDataModel = cr.ui.ArrayDataModel;

  function SwitcherManageOptions() {
    OptionsPage.call(this, 'switcherManage', '', 'switcherManagePage');
  }

  cr.addSingletonGetter(SwitcherManageOptions);

  SwitcherManageOptions.prototype = {
    __proto__: options.OptionsPage.prototype,

    initializePage: function() {
      OptionsPage.prototype.initializePage.call(this);
      var $list = $("switcherManageList");
      SwitcherList.decorate($list);

      window.getHandleGetAllSwitchesCallback = function(list) {
        $list.setExceptions(list);
      };
      window.importSwitchFileCallback = function(count) {
        chrome.send('getAllSwitches');
      };

      $('importSwitcherListButton').onclick = function() {
        chrome.send('importSwitchFile');
      };
      $('exportSwitcherListButton').onclick = function() {
        chrome.send('exportSwitchFile');
      };
    },
    didShowPage: function() {
      if (!this.showed) {
        this.showed = true;
        chrome.send('getAllSwitches');
      }
    }
  };

  var SwitcherList = cr.ui.define('list');

  SwitcherList.prototype = {
    __proto__: InlineEditableItemList.prototype,

    /**
     * Called when an element is decorated as a list.
     */
    decorate: function() {
      InlineEditableItemList.prototype.decorate.call(this);

      this.classList.add('settings-list');

      for (var parentNode = this.parentNode; parentNode; parentNode = parentNode.parentNode) {
        if (parentNode.hasAttribute('contentType')) {
          this.contentType = parentNode.getAttribute('contentType');
          break;
        }
      }

      this.mode = this.getAttribute('mode');

      var exceptionList = this;

      // Whether the exceptions in this list allow an 'Ask every time' option.
      this.enableAskOption = this.contentType == 'plugins';

      this.autoExpands = true;
      this.reset();
    },

    /**
     * Creates an item to go in the list.
     * @param {Object} entry The element from the data model for this row.
     */
    createItem: function(entry) {
      if (entry) {
        return new ExceptionsListItem(this.contentType,
          this.mode,
          this.enableAskOption,
          entry);
      } else {
        var addRowItem = new ExceptionsAddRowListItem(this.contentType,
          this.mode,
          this.enableAskOption);
        addRowItem.deletable = false;
        return addRowItem;
      }
    },

    /**
     * Sets the exceptions in the js model.
     * @param {Object} entries A list of dictionaries of values, each dictionary
     *     represents an exception.
     */
    setExceptions: function(entries) {
      var deleteCount = this.dataModel.length;

      if (this.isEditable()) {
        // We don't want to remove the Add New Exception row.
        deleteCount = deleteCount - 1;
      }

      var args = [0, deleteCount];
      args.push.apply(args, entries);
      this.dataModel.splice.apply(this.dataModel, args);
    },

    /**
     * The browser has finished checking a pattern for validity. Update the
     * list item to reflect this.
     * @param {string} pattern The pattern.
     * @param {bool} valid Whether said pattern is valid in the context of
     *     a content exception setting.
     */
    patternValidityCheckComplete: function(pattern, valid) {
      var listItems = this.items;
      for (var i = 0; i < listItems.length; i++) {
        var listItem = listItems[i];
        // Don't do anything for messages for the item if it is not the intended
        // recipient, or if the response is stale (i.e. the input value has
        // changed since we sent the request to analyze it).
        if (pattern == listItem.input.value)
          listItem.setPatternValid(valid);
      }
    },

    /**
     * Returns whether the rows are editable in this list.
     */
    isEditable: function() {
      // Editing notifications and geolocation is disabled for now.
      return !(this.contentType == 'notifications' ||
        this.contentType == 'location' ||
        this.contentType == 'fullscreen' ||
        this.contentType == 'media-stream');
    },

    /**
     * Removes all exceptions from the js model.
     */
    reset: function() {
      if (this.isEditable()) {
        // The null creates the Add New Exception row.
        this.dataModel = new ArrayDataModel([null]);
      } else {
        this.dataModel = new ArrayDataModel([]);
      }
    },

    /** @inheritDoc */
    deleteItemAtIndex: function(index) {
      var listItem = this.getListItemByIndex(index);
      if (listItem.undeletable)
        return;

      chrome.send('removeSwitch', [listItem.pattern]);
      chrome.send('getAllSwitches');
    },
  };

  function ExceptionsListItem(contentType, mode, enableAskOption, exception) {
    var el = cr.doc.createElement('div');
    el.mode = mode;
    el.contentType = contentType;
    el.enableAskOption = enableAskOption;
    el.dataItem = exception;
    el.__proto__ = ExceptionsListItem.prototype;
    el.decorate();

    return el;
  }

  ExceptionsListItem.prototype = {
    __proto__: InlineEditableItem.prototype,

    /**
     * Called when an element is decorated as a list item.
     */
    decorate: function() {
      InlineEditableItem.prototype.decorate.call(this);

      this.isPlaceholder = !this.pattern;
      var patternCell = this.createEditableTextCell(this.pattern);
      patternCell.className = 'exception-pattern';
      patternCell.classList.add('weakrtl');
      this.contentElement.appendChild(patternCell);
      if (this.pattern)
        this.patternLabel = patternCell.querySelector('.static-text');
      var input = patternCell.querySelector('input');

      // TODO(stuartmorgan): Create an createEditableSelectCell abstracting
      // this code.
      // Setting label for display mode. |pattern| will be null for the 'add new
      // exception' row.
      if (this.pattern) {
        var settingLabel = cr.doc.createElement('span');
        settingLabel.textContent = this.settingForDisplay();
        settingLabel.className = 'exception-setting';
        settingLabel.setAttribute('displaymode', 'static');
        this.contentElement.appendChild(settingLabel);
        this.settingLabel = settingLabel;
      }

      // Setting select element for edit mode.
      var select = cr.doc.createElement('select');

      [{
        text: 'WebKit',
        value: 'WebKit'
      }, {
        text: 'IE',
        value: 'IE'
      }, {
        text: 'IE 7',
        value: 'IE 7'
      }, {
        text: 'IE 8',
        value: 'IE 8'
      }, {
        text: 'IE 9',
        value: 'IE 9'
      }, {
        text: 'IE 10',
        value: 'IE 10'
      }, {
        text: 'IE 11',
        value: 'IE 11'
      }].forEach(function(item) {
        var option = cr.doc.createElement('option');
        option.textContent = item.text;
        option.value = item.value;
        select.appendChild(option);
      });

      this.contentElement.appendChild(select);
      this.editFields_.push(select);
      select.className = 'exception-setting';
      if (this.pattern)
        select.setAttribute('displaymode', 'edit');
      if (this.contentType == 'media-stream') {
        this.settingLabel.classList.add('media-audio-setting');

        var videoSettingLabel = cr.doc.createElement('span');
        videoSettingLabel.textContent = this.videoSettingForDisplay();
        videoSettingLabel.className = 'exception-setting';
        videoSettingLabel.classList.add('media-video-setting');
        videoSettingLabel.setAttribute('displaymode', 'static');
        this.contentElement.appendChild(videoSettingLabel);
      }

      // Used to track whether the URL pattern in the input is valid.
      // This will be true if the browser process has informed us that the
      // current text in the input is valid. Changing the text resets this to
      // false, and getting a response from the browser sets it back to true.
      // It starts off as false for empty string (new exceptions) or true for
      // already-existing exceptions (which we assume are valid).
      this.inputValidityKnown = this.pattern;
      // This one tracks the actual validity of the pattern in the input. This
      // starts off as true so as not to annoy the user when he adds a new and
      // empty input.
      this.inputIsValid = true;

      this.input = input;
      this.select = select;

      this.updateEditables();

      // Editing notifications, geolocation and media-stream is disabled for
      // now.
      if (this.contentType == 'notifications' ||
        this.contentType == 'location' ||
        this.contentType == 'media-stream') {
        this.editable = false;
      }

      // If the source of the content setting exception is not a user
      // preference, that source controls the exception and the user cannot edit
      // or delete it.
      var controlledBy =
        this.dataItem.source && this.dataItem.source != 'preference' ?
        this.dataItem.source : null;

      if (controlledBy) {
        this.setAttribute('managedby', controlledBy);
        this.deletable = false;
        this.editable = false;
      }

      // Listen for edit events.
      this.addEventListener('canceledit', this.onEditCancelled_);
      this.addEventListener('commitedit', this.onEditCommitted_);
    },

    /**
     * The pattern (e.g., a URL) for the exception.
     * @type {string}
     */
    get pattern() {
      return this.dataItem['origin'];
    },
    set pattern(pattern) {
      this.dataItem['origin'] = pattern;
    },

    /**
     * The setting (allow/block) for the exception.
     * @type {string}
     */
    get setting() {
      return this.dataItem['setting'];
    },
    set setting(setting) {
      this.dataItem['setting'] = setting;
    },

    /**
     * Gets a human-readable setting string.
     * @type {string}
     */
    settingForDisplay: function() {
      var setting = this.setting;
      var map = {
        'WebKit': 'WebKit',
        'IE': 'IE',
        'IE 7': 'IE 7',
        'IE 8': 'IE 8',
        'IE 9': 'IE 9',
        'IE 10': 'IE 10',
        'IE 11': 'IE 11'
      };
      return map[setting] || '';
    },

    /**
     * media video specific function.
     * Gets a human-readable video setting string.
     *
     * @return {string} The display string.
     */
    videoSettingForDisplay: function() {
      return this.getDisplayStringForSetting(this.dataItem.video);
    },

    /**
     * Gets a human-readable display string for setting.
     *
     * @param {string} setting The setting to be displayed.
     * @return {string} The display string.
     */
    getDisplayStringForSetting: function(setting) {
      if (setting == 'allow' || setting == 'block' || setting == 'ask' || setting == 'session')
        return templateData[setting + 'Exception'];
      else if (setting == 'default')
        return '';

      console.error('Unknown setting: [' + setting + ']');
      return '';
    },

    /**
     * Update this list item to reflect whether the input is a valid pattern.
     * @param {boolean} valid Whether said pattern is valid in the context of
     *     a content exception setting.
     */
    setPatternValid: function(valid) {
      if (valid || !this.input.value)
        this.input.setCustomValidity('');
      else
        this.input.setCustomValidity(' ');
      this.inputIsValid = valid;
      this.inputValidityKnown = true;
    },

    /**
     * Set the <input> to its original contents. Used when the user quits
     * editing.
     */
    resetInput: function() {
      this.input.value = this.pattern;
    },

    /**
     * Copy the data model values to the editable nodes.
     */
    updateEditables: function() {
      this.resetInput();

      var settingOption =
        this.select.querySelector('[value=\'' + this.setting + '\']');
      if (settingOption)
        settingOption.selected = true;
    },

    /** @inheritDoc */
    get currentInputIsValid() {
      // return this.inputValidityKnown && this.inputIsValid;
      return this.input.value != '' && this.inputIsValid;
    },

    /** @inheritDoc */
    get hasBeenEdited() {
      var livePattern = this.input.value;
      var liveSetting = this.select.value;
      return livePattern != this.pattern || liveSetting != this.setting;
    },

    /**
     * Called when committing an edit.
     * @param {Event} e The end event.
     * @private
     */
    onEditCommitted_: function(e) {
      var newPattern = this.input.value;
      var newSetting = this.select.value;

      this.finishEdit(newPattern, newSetting);
    },

    /**
     * Called when cancelling an edit; resets the control states.
     * @param {Event} e The cancel event.
     * @private
     */
    onEditCancelled_: function() {
      this.updateEditables();
      this.setPatternValid(true);
    },

    /**
     * Editing is complete; update the model.
     * @param {string} newPattern The pattern that the user entered.
     * @param {string} newSetting The setting the user chose.
     */
    finishEdit: function(newPattern, newSetting) {
      this.patternLabel.textContent = newPattern;
      this.settingLabel.textContent = this.settingForDisplay();
      var oldPattern = this.pattern;
      this.pattern = newPattern;
      this.setting = newSetting;

      // TODO(estade): this will need to be updated if geolocation/notifications
      // become editable.
      if (oldPattern != newPattern) {
        chrome.send('removeSwitch', [oldPattern]);
      }

      chrome.send('updateSwitch', [newPattern, newSetting]);
      chrome.send('getAllSwitches');
    }
  };

  function ExceptionsAddRowListItem(contentType, mode, enableAskOption) {
    var el = cr.doc.createElement('div');
    el.mode = mode;
    el.contentType = contentType;
    el.enableAskOption = enableAskOption;
    el.dataItem = [];
    el.__proto__ = ExceptionsAddRowListItem.prototype;
    el.decorate();

    return el;
  }

  ExceptionsAddRowListItem.prototype = {
    __proto__: ExceptionsListItem.prototype,

    decorate: function() {
      ExceptionsListItem.prototype.decorate.call(this);

      this.input.placeholder = localStrings.getString('IDS_OPTIONS_KERNAL_SWITCH_SETTING_TIPS');

      // Do we always want a default of allow?
      this.setting = 'allow';
    },

    /**
     * Clear the <input> and let the placeholder text show again.
     */
    resetInput: function() {
      this.input.value = '';
    },

    /** @inheritDoc */
    get hasBeenEdited() {
      return this.input.value != '';
    },

    /**
     * Editing is complete; update the model. As long as the pattern isn't
     * empty, we'll just add it.
     * @param {string} newPattern The pattern that the user entered.
     * @param {string} newSetting The setting the user chose.
     */
    finishEdit: function(newPattern, newSetting) {
      this.resetInput();
      chrome.send('updateSwitch', [newPattern, newSetting]);
      chrome.send('getAllSwitches');
    },
  };

  return {
    SwitcherManageOptions: SwitcherManageOptions
  };
});
